<?php
/**
 * Plugin Name: LoftOcean REST API Enabler
 * Description: Enables REST API access for LoftOcean Rooms and Bookings, and exposes all meta fields.
 * Version: 1.3
 * Author: Jessy Ledama
 */
 
 add_action('init', function () {
    $meta_fields = ['room_id', 'room_name', 'checkin', 'checkout', 'source_title'];

    foreach ($meta_fields as $field) {
        register_post_meta('loftocean_bookings', $field, [
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'string',
            'auth_callback' => function() { return true; },
        ]);
    }
});

add_action('wp_loaded', function() {
    $post_types = ['loftocean_room', 'loftocean_bookings']; // both post types
    global $wp_post_types;

    foreach ($post_types as $post_type) {
        if (isset($wp_post_types[$post_type])) {
            $wp_post_types[$post_type]->show_in_rest = true;
            $wp_post_types[$post_type]->rest_base = $post_type;
            $wp_post_types[$post_type]->rest_controller_class = 'WP_REST_Posts_Controller';

            // Expose all meta fields for this post type
            register_rest_field($post_type, 'meta', [
                'get_callback' => function ($object) {
                    $meta = get_post_meta($object['id']);
                    $clean_meta = [];
                    foreach ($meta as $key => $value) {
                        $clean_meta[$key] = maybe_unserialize($value[0]);
                    }
                    return $clean_meta;
                },
                
                'update_callback' => function ($value, $object) {
                    // Allow updating meta fields via REST API
                    if (is_array($value)) {
                        foreach ($value as $key => $val) {
                            if (!empty($key)) {
                                update_post_meta($object->ID, $key, maybe_serialize($val));
                            }
                        }
                    }
                },
                'schema' => null,
            ]);
        }
    }

    // Optional debug: view all meta for a specific post
    if (current_user_can('administrator') && isset($_GET['debug_post_meta'])) {
        $post_id = (int) $_GET['debug_post_meta'];
        $meta = get_post_meta($post_id);
        echo '<pre>';
        print_r($meta);
        echo '</pre>';
        exit;
    }
});


add_filter('rest_loftocean_bookings_query', function ($args, $request) {
        $args['post_status'] = ['private', 'publish', 'checked_in'];
        return $args;
    }, 10, 2);
    
add_action('rest_api_init', function () {
    register_rest_route('custom/v1', '/bookings', [
        'methods' => 'GET',
        'callback' => 'custom_list_bookings',
        'permission_callback' => '__return_true',
    ]);
    
    register_rest_route('custom/v1', '/create-bookings', [
        'methods' => 'POST',
        'callback' => 'custom_create_booking',
        'permission_callback' => '__return_true',
    ]);
    
    register_rest_route('custom/v1', '/cancel-booking', [
        'methods' => 'POST',
        'callback' => 'custom_cancel_booking',
        'permission_callback' => '__return_true',
    ]);
});

function custom_list_bookings(WP_REST_Request $request) {
    $query = new WP_Query([
        'post_type'      => 'loftocean_bookings',
        'post_status'    => ['private', 'publish'],
        'posts_per_page' => -1,
    ]);

    $bookings = [];

    foreach ($query->posts as $post) {
        $bookings[] = [
            'id'     => $post->ID,
            'title'  => $post->post_title,
            'status' => $post->post_status,
            'meta'   => get_post_meta($post->ID),
        ];
    }

    return $bookings;
}

function custom_cancel_booking(WP_REST_Request $request) {
    global $wpdb;

    $params = $request->get_json_params();

    $room_id  = intval($params['room_id'] ?? 0);
    $checkin  = sanitize_text_field($params['checkin'] ?? '');
    $checkout = sanitize_text_field($params['checkout'] ?? '');
    $booking_id = intval($params['booking_id'] ?? 0);

    if (!$room_id || empty($checkin) || empty($checkout)) {
        return new WP_Error('missing_params', 'room_id, checkin and checkout are required', ['status' => 400]);
    }

    // Convert to timestamps if needed
    $checkin_ts  = is_numeric($checkin) ? intval($checkin) : strtotime($checkin);
    $checkout_ts = is_numeric($checkout) ? intval($checkout) : strtotime($checkout);

    $custom_table = $wpdb->prefix . 'loftocean_room_custom_data';
    $order_table  = $wpdb->prefix . 'loftocean_room_order';

    // ✅ 1. Update availability table — mark room as available
    $wpdb->query(
        $wpdb->prepare(
            "UPDATE $custom_table 
             SET status = %s, number_booked = 0
             WHERE room_id = %d 
             AND checkin >= %d 
             AND checkout <= %d",
            'available', $room_id, $checkin_ts, $checkout_ts
        )
    );

    // ✅ 2. Update room order table — mark booking as not booked
    $wpdb->query(
        $wpdb->prepare(
            "UPDATE $order_table 
             SET booked = 0 
             WHERE checkin = %d 
             AND checkout = %d",
            $checkin_ts, $checkout_ts
        )
    );

    // ✅ 3. Optionally update booking post status
    if ($booking_id) {
        wp_update_post([
            'ID' => $booking_id,
            'post_status' => 'cancelled', // you can define this custom status
        ]);
    }

    return new WP_REST_Response([
        'success' => true,
        'message' => "Booking cancelled and room made available again.",
        'data' => [
            'room_id' => $room_id,
            'checkin' => $checkin_ts,
            'checkout' => $checkout_ts,
        ]
    ], 200);
}


function custom_create_booking(WP_REST_Request $request) {
    global $wpdb;

    $params = $request->get_json_params();

    $room_id      = intval($params['room_id'] ?? 0);
    $room_name    = sanitize_text_field($params['room_name'] ?? '');
    $checkin      = sanitize_text_field($params['checkin'] ?? '');
    $checkout     = sanitize_text_field($params['checkout'] ?? '');
    $status       = sanitize_text_field($params['status'] ?? 'private');
    $source_title = sanitize_text_field($params['source_title'] ?? 'Odoo');
    $title        = sanitize_text_field($params['title'] ?? "Booking for Room {$room_id}");
    
    // 🔍 If room_name is provided but no room_id, try to find the room post
    if (empty($room_id) && !empty($room_name)) {
        $room_query = new WP_Query([
            'post_type'      => 'loftocean_room',
            'post_status'    => ['publish', 'private'],
            'title'          => $room_name,
            'posts_per_page' => 1,
        ]);

        if (!empty($room_query->posts)) {
            $room_id = $room_query->posts[0]->ID;
        } else {
            return new WP_Error(
                'room_not_found',
                "No room found matching name '{$room_name}'",
                ['status' => 404]
            );
        }
    }

    if (!$room_id) {
        return new WP_Error('missing_room', 'Room ID is required', ['status' => 400]);
    }


    // ✅ 1. Create booking post
    $booking_id = wp_insert_post([
        'post_type'   => 'loftocean_bookings',
        'post_title'  => $title,
        'post_status' => $status,
    ]);

    if (is_wp_error($booking_id)) {
        return new WP_Error('insert_failed', 'Could not create booking', ['status' => 500]);
    }
    

    // ✅ 2. Save post meta
    update_post_meta($booking_id, 'room_id', $room_id);
    update_post_meta($booking_id, 'room_name', $room_name);
    update_post_meta($booking_id, 'checkin', $checkin);
    update_post_meta($booking_id, 'checkout', $checkout);
    update_post_meta($booking_id, 'source_title', $source_title);
    
    // 🔍 Get the room_unique_id from relationship table
    $relationship_table = $wpdb->prefix . 'loftocean_room_relationship';
    $room_unique_id = $wpdb->get_var(
        $wpdb->prepare("SELECT room_unique_id FROM $relationship_table WHERE room_id = %d LIMIT 1", $room_id)
    );

    // ✅ 3. Insert into loftocean_room_order
    $order_table = $wpdb->prefix . 'loftocean_room_order';
    
    $checkin_ts  = is_numeric($checkin) ? intval($checkin) : strtotime($checkin);
    $checkout_ts = is_numeric($checkout) ? intval($checkout) : strtotime($checkout);
    
    $wpdb->insert(
        $order_table,
        [
            'room_unique_id' => $room_unique_id,
            'checkin'   => $checkin_ts,
            'checkout'  => $checkout_ts,
            'booked' => "1"
        ],
        ['%s', '%d', '%d', '%d']
    );

    // ✅ 4. Insert into loftocean_room_custom_data for each booked date
    $custom_table = $wpdb->prefix . 'loftocean_room_custom_data';

    $checkin_ts  = is_numeric($checkin) ? intval($checkin) : strtotime($checkin);
    $checkout_ts = is_numeric($checkout) ? intval($checkout) : strtotime($checkout);


    $wpdb->insert(
        $custom_table,
        [
            'room_id'          => $room_id,
            'checkin'          => $checkin_ts,
            'checkout'         => $checkout_ts,
            'status'           => 'unavailable',
            'number_booked'    => 1,
        ],
        ['%d', '%d', '%d', '%s', '%d']
    );
    $current_day = strtotime('+1 day', $current_day);

    // ✅ Return response
    return new WP_REST_Response([
        'success'     => true,
        'booking_id'  => $booking_id,
        'message'     => 'Booking created successfully and all tables updated',
        'data'        => [
            'room_id'   => $room_id,
            'checkin'   => $checkin,
            'checkout'  => $checkout,
            'room_name' => $room_name,
        ],
    ], 200);
}



function custom_ical_sync_schedules( $schedules ) {
    // Add a 'tenminutes' schedule
    $schedules['tenminutes'] = array(
        'interval' => 10 * 60, // 10 minutes in seconds
        'display'  => __( 'Every 10 minutes', 'textdomain' )
    );

    // Add a 'biweekly' schedule
    $schedules['biweekly'] = array(
        'interval' => 14 * 24 * 60 * 60, // 2 weeks in seconds
        'display'  => __( 'Every two weeks', 'textdomain' )
    );

    return $schedules;
}
add_filter( 'cron_schedules', 'custom_ical_sync_schedules' );
